### Полное техническое задание: Cyberspace Knowledge Management System

**Цель проекта**: Создать самоустанавливающуюся платформу для управления знаниями и файлами с веб-интерфейсом и WebDAV-доступом, разворачиваемую простым копированием файлов в корень веб-сервера.

---

### 1. Системные требования
- **Веб-сервер**: Apache 2.4+ (mod_rewrite включен)
- **PHP**: 8.3+ (модули: pdo_mysql, gd, mbstring, fileinfo, zip)
- **База данных**: MySQL 8.0+ или MariaDB 10.6+
- **Клиентская часть**: Vanilla JS (ES6+), CSS3
- **Дисковое пространство**: 100МБ+ (мин. для установки)

---

### 2. Структура проекта
```
cyberspace/                      # DocumentRoot веб-сервера
├── .htaccess                    # Правила безопасности и маршрутизации
├── index.php                    # Главный контроллер
├── install.php                  # Установщик системы
├── locale/                  
│   ├── en.json              # {"auth": {"login": "Sign In"}, ...}
│   └── ru.json              # Группировка по функциональным блокам
├── config/                      # Конфигурация (защищённый доступ)
│   ├── app.php                 # Основные настройки
│   ├── database.php            # Данные подключения к БД
│   ├── plugins.php            # Активные плагины
│   ├── mime_types.php          # Разрешенные типы файлов
│   └── environment.php         # Параметры среды
├── data/                        # Пользовательские данные
│   ├── public/                  # Публичные ресурсы (симлинки)
│   ├── user/                    # Данные пользователей
│   │   └── {username}/          # Директория пользователя
│   │       ├── files/           # Основные файлы
│   │       ├── images/          # Изображения для редактора
│   │       └── .meta/           # Метаданные (версии, блокировки)
│   └── temp/                    # Временные файлы
├── logs/                        # Логи приложения
│   └── app.log                  # Основной лог-файл
├── plugins/                     # Плагины
│   ├── core_auth/               # Аутентификация (ядро)
│   ├── filemanager/             # Файловый менеджер
│   ├── markdown_editor/         # Редактор Markdown
│   ├── admin_panel/             # Административная панель
│   ├── webdav/                  # WebDAV интеграция
│   ├── taxonomy/                # Система таксономий
│   └── sharing/                 # Система общего доступа
├── public/                      # Публичные ресурсы
│   ├── assets/                  # Глобальные ресурсы
│   └── plugins/                 # Ресурсы плагинов (симлинки)
├── src/                         # Исходный код ядра
│   ├── Core/                    # Основные компоненты
│   ├── Models/                  # Модели данных
│   ├── Services/                # Сервисы ядра
│   └── Utilities/               # Вспомогательные утилиты
├── templates/                   # Шаблоны ядра
│   ├── layouts/                 # Макеты страниц
│   ├── partials/                # Частичные шаблоны
│   └── pages/                   # Шаблоны конкретных страниц
└── vendor/                      # Зависимости Composer (опционально)
```

---

### 3. Функциональные требования

#### 3.1 Установка и настройка
- **Автоматический запуск** при первом обращении
- **Пошаговый процесс**:
  1. Проверка требований сервера (PHP, модули)
  2. Настройка подключения к БД
  3. Создание таблиц в БД
  4. Создание учётной записи администратора
  5. Активация основных плагинов
- **Создание файла блокировки** после успешной установки

#### 3.2 Аутентификация и пользователи
- **Регистрация** новых пользователей
- **Вход по логину/паролю**
- **Восстановление пароля** через email
- **Ролевая модель**:
  - Гость (только просмотр публичного контента)
  - Пользователь (базовые права)
  - Редактор (расширенные права редактирования)
  - Администратор (полный доступ)
- **Профиль пользователя** с возможностью смены пароля

#### 3.3 Главная страница
- **Двухколоночный макет**:
  ```html
  <div class="home-container">
    <div class="readme-panel">
      <!-- Содержимое /data/README.md -->
    </div>
    <div class="public-directories">
      <!-- Список публичных каталогов -->
    </div>
  </div>
  ```
- **Автоматический парсинг** README.md (Markdown → HTML)
- **Динамическое обновление** списка публичных каталогов
- **Навигация** по каталогам с хлебными крошками

#### 3.4 Файловая система
- **Личное пространство** для каждого пользователя: `/data/user/{username}`
- **Публичные файлы/папки** (доступны гостям)
- **Квотирование дискового пространства**
- **Ограничение типов файлов** (настраивается в админке)
- **Конфликты имён**:
  - Автоматическое переименование (file.txt → file_1.txt)
  - Форма выбора действия при загрузке:
    ```json
    {
      "options": ["Заменить", "Автопереименование", "Указать имя вручную"]
    }
    ```

#### 3.5 Файловый менеджер
- **Иерархическое представление** файлов и папок
- **Операции**:
  - Загрузка файлов (с прогресс-баром)
  - Создание папок
  - Переименование
  - Удаление
  - Копирование/перемещение
- **Контекстное меню** для файлов/папок
- **Управление публичностью** ресурсов

#### 3.6 Редактор Markdown
- **Режимы редактирования**:
  - Только текст
  - Предпросмотр
  - Двухколоночный режим (редактор + предпросмотр)
- **Панель инструментов** для форматирования
- **Загрузка изображений** в подкаталог `images`
- **Автосохранение** каждые 60 секунд
- **Версионирование**:
  - Визуальное сравнение версий
  - Возможность восстановления
  - При редактировании устаревшей версии - сохранение как новой версии:
    ```mermaid
    sequenceDiagram
      Пользователь->>Система: Открывает версию 3 для редактирования
      Система->>Пользователь: Показывает diff (v3 vs v5)
      Пользователь->>Система: Сохраняет изменения
      Система->>БД: Создаёт версию 6
    ```

#### 3.7 Система общего доступа
- **Группы пользователей**:
  - Создание групп (любым пользователем)
  - Приглашение участников
  - Роли: участник, администратор группы
- **Общий доступ к ресурсам**:
  - Конкретным пользователям
  - Группам пользователей
  - Публичная ссылка (с ограничением срока действия)
- **Уровни доступа**:
  - Просмотр
  - Редактирование
  - Управление (изменение прав)
- **Интерфейс управления доступом** в контекстном меню

#### 3.8 Административная панель
- **Управление пользователями**:
  - Просмотр списка
  - Создание/редактирование
  - Смена ролей
- **Управление плагинами**:
  - Активация/деактивация
  - Настройка
- **Системные настройки**:
  - Разрешенные MIME-типы (интерфейс):
    ```php
    // AdminController
    public function manageMimeTypes() {
      $types = ['image/jpeg', 'application/pdf', ...];
      // Интерфейс для добавления/удаления
    }
    ```
  - Настройки квот
  - Настройки почты
- **Просмотр системных логов**
- **Мониторинг ресурсов** сервера

#### 3.9 WebDAV интеграция
- **Доступ к личному пространству** через `/webdav`
- **Аутентификация** через систему
- **Поддержка блокировок** (LOCK/UNLOCK)
- **Совместимость** с клиентами (Windows Explorer, macOS Finder, Linux)

---

### 4. Конфликты и решения

#### 4.1 Одновременное редактирование файлов
- **Проблема**: Несколько пользователей редактируют один файл
- **Решение**:
  - **Блокировки**: Система временных файлов блокировки
  - **Визуальное уведомление** в интерфейсе
  - **Принудительная синхронизация** при сохранении
  ```php
  class FileLockService {
      public function acquireLock(int $fileId, int $userId): bool {
          // Проверка существующей блокировки
          // Создание lock-файла с метаданными
      }
      
      public function releaseLock(int $fileId): void {
          // Удаление lock-файла
      }
  }
  ```

#### 4.2 Конфликт версий при редактировании
- **Проблема**: Редактирование устаревшей версии
- **Решение**:
  - **Визуальное сравнение** версий (diff)
  - **Сохранение как новой версии** с сохранением истории
  - **Интерфейс выбора**:
    ```javascript
    function showVersionConflict(currentVer, serverVer) {
      renderDiff(currentVer.content, serverVer.content);
      showOptions([
        "Сохранить как новую версию (v" + (serverVer.number + 1) + ")",
        "Перезагрузить текущую версию",
        "Отменить редактирование"
      ]);
    }
    ```

#### 4.3 Конфликт прав доступа
- **Проблема**: Противоречивые разрешения для групп/пользователей
- **Решение**:
  - **Приоритет индивидуальных прав** над групповыми
  - **Визуальное предупреждение** в интерфейсе
  ```php
  class PermissionResolver {
      public function resolve(User $user, Resource $resource): string {
          // 1. Проверка индивидуальных прав
          // 2. Проверка групповых прав
          // 3. Применение приоритетов
      }
  }
  ```

---

### 5. Безопасность

1. **Защита данных**:
   - Экранирование вывода (XSS защита)
   - Подготовленные SQL-запросы
   - CSRF-токены во всех формах

2. **Аутентификация**:
   - Bcrypt хеширование паролей
   - Сессионная фиксация
   - Ограничение попыток входа

3. **Файловая система**:
   - Проверка MIME-типов при загрузке
   - Санитизация имен файлов
   - Ограничение пути (path traversal protection)

4. **Доступ**:
   - Проверка прав для всех операций
   - Изоляция пользовательских данных

---

### 6. Схема базы данных (ключевые таблицы)

```mermaid
erDiagram
    users ||--o{ files : "1:N"
    users ||--o{ groups : "владелец"
    users ||--o{ group_members : "участник"
    groups ||--o{ group_members : "1:N"
    files ||--o{ file_versions : "1:N"
    files ||--o{ file_taxonomy : "1:N"
    taxonomies ||--o{ taxonomy_terms : "1:N"
    taxonomy_terms ||--o{ file_taxonomy : "1:N"
    files ||--o{ shared_resources : "1:N"
    groups ||--o{ shared_permissions : "1:N"
    users ||--o{ shared_permissions : "1:N"
    files ||--o{ file_locks : "1:N"
    users ||--o{ file_locks : "1:N"
    
    users {
        int id PK
        varchar(50) username
        varchar(100) email
        varchar(255) password_hash
        enum("guest","user","editor","admin") role
        bigint disk_quota
        bigint disk_used
        datetime created_at
        datetime updated_at
    }
    
    files {
        int id PK
        int user_id FK
        varchar(512) path
        varchar(255) name
        bigint size
        varchar(100) mime_type
        boolean is_directory
        boolean is_public
        datetime created_at
        datetime updated_at
    }
    
    file_versions {
        int id PK
        int file_id FK
        text content
        int version_number
        int user_id FK
        datetime created_at
    }
    
    groups {
        int id PK
        int owner_id FK
        varchar(100) name
        text description
        boolean is_public
        datetime created_at
    }
    
    group_members {
        int group_id FK
        int user_id FK
        enum("member","admin") role
    }
    
    shared_permissions {
        int id PK
        int resource_id FK
        enum("file","folder") resource_type
        int target_id FK
        enum("user","group") target_type
        enum("view","edit","manage") permission_level
        boolean can_share
        datetime expires_at
    }
    
    taxonomies {
        int id PK
        varchar(50) name
        varchar(50) slug
        text description
    }
    
    taxonomy_terms {
        int id PK
        int taxonomy_id FK
        varchar(100) name
        varchar(100) slug
        int parent_id FK
    }
    
    file_taxonomy {
        int file_id FK
        int term_id FK
    }
    
    file_locks {
        int file_id FK
        int user_id FK
        datetime locked_at
        datetime expires_at
    }
```

---

### 7. Процесс установки

1. **Копирование файлов**:
   ```bash
   scp -r cyberspace/ user@server:/var/www/html/
   ```

2. **Настройка веб-сервера** (пример для Apache):
   ```apache
   <VirtualHost *:80>
       ServerName devel.cyberwide.ru
       DocumentRoot /var/www/html/cyberspace
       <Directory /var/www/html/cyberspace>
           AllowOverride All
           Require all granted
       </Directory>
   </VirtualHost>
   ```

3. **Первоначальный запуск**:
   - Открыть http://devel.cyberwide.ru в браузере
   - Следовать инструкциям установщика

4. **Последовательность установки**:
   1. Проверка требований сервера
   2. Настройка подключения к БД
   3. Создание таблиц
   4. Создание администратора
   5. Настройка основных параметров
   6. Активация плагинов

---

### 8. Гарантии полноты реализации

Учтены все требования из диалога:
- Парсинг README.md на главной странице
- Отображение публичных каталогов в двухколоночном формате
- Система общего доступа с группами пользователей
- Настройка разрешенных типов файлов в админке
- Конфликт-менеджмент для:
  - Одновременного редактирования
  - Конфликта имен файлов
  - Версионных конфликтов
- Визуальное сравнение версий перед редактированием
- Варианты действий при конфликте имен:
  - Замена
  - Автопереименование
  - Ручное указание имени
- Сохранение редактируемой версии как новой независимо от истории

Данное ТЗ полностью описывает систему без необходимости дополнительных уточнений. Разработка может начинаться немедленно.